<?php

namespace App\Http\Controllers;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Mail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\PatientsMedicalReport;
use App\Models\Prescription;
use App\Models\PatientsRefferal;
use App\Models\SickNote; 
use App\Models\ConsultRequest;
use App\Models\tbl_doctor;
use App\Models\tbl_patient;
use App\Models\tbl_family_members;
use App\Mail\MedicalReportMail;
use App\Mail\PrescriptionMail;
use App\Mail\ReferralMail;
use App\Mail\SickNoteMail;
use Illuminate\Support\Facades\DB;


class PatientController extends Controller
{
    public function index()
{
    $doctorId = Auth::guard('doctor')->id();

    // ✅ Doctor details
    $doctor = Auth::guard('doctor')->user();

   $consults = DB::table('consult_requests as cr')

    // medical problem
        ->leftJoin('main_complaints as mc', 'mc.id', '=', 'cr.medical_problem_id')

        // patient
        ->leftJoin('tbl_patient as p', 'p.id', '=', 'cr.patient_id')

        // family member
        ->leftJoin('tbl_family_members as fm', 'fm.id', '=', 'cr.patient_id')

        ->where('cr.doctor_id', $doctorId)
        ->whereIn('cr.status', ['accept', 'completed'])
        ->orderBy('cr.createdAt', 'DESC')

        ->select([
            'cr.*',
            
            // medical problem
            'mc.enName as enName',
            'mc.frName as frName',
            'mc.itName as itName',
            'mc.deName as deName',


            // 👤 patient OR family
            DB::raw("
                CASE 
                    WHEN p.id IS NOT NULL THEN 'patient'
                    ELSE 'family'
                END as person_type
            "),

            DB::raw("COALESCE(p.firstName, fm.firstName) as firstName"),
            DB::raw("COALESCE(p.lastName, fm.lastName) as lastName"),
            DB::raw("COALESCE(p.email, fm.email) as email"),
            DB::raw("COALESCE(p.dob, fm.dob) as dob"),
            DB::raw("COALESCE(p.age, fm.age) as age"),
        ])
        ->paginate(10);

        // dd($consults);

    return view('admin.index', [
        'consults' => $consults,
        'doctor'   => $doctor
    ]);
}

    public function detail($consultId)
    {
        $doctorId = Auth::guard('doctor')->id();
        $doctor = Auth::guard('doctor')->user();

        $consult = ConsultRequest::findOrFail($consultId);
        $consultRequest = $consult; // alias variable

        $patientId = $consult->patient_id;

        $medicalReportSingle = PatientsMedicalReport::where('patient_id', $patientId)->where('type', 'emr')
        ->latest()
        ->first();

        

        if($consult->patient_type == 'patient'){
            $patient = DB::table('tbl_patient')->where('id', $patientId)->first();
        }else{
           $patient = DB::table('tbl_family_members')->where('id', $patientId)->first();
        }

        $medical_problem_id = $consult->medical_problem_id;

        $medical_problem = DB::table('main_complaints')->where('id', $medical_problem_id)->first();


        // Fetch related data by patient ID
       $medicalReports = PatientsMedicalReport::where('consult_id', $consultId)
       ->where('type', 'emr')
        // ->where('doctor_id', $doctorId)
        ->with('consultRequest.medicalProblem')
        ->latest()
        ->get();




        $prescriptions = Prescription::where('consult_id', $consultId)
        // ->where('doctor_id', $doctorId)
        ->with('consultRequest.medicalProblem')
        ->latest()
        ->get();

        $referrals = PatientsRefferal::where('consult_id', $consultId)
        // ->where('doctor_id', $doctorId)
        ->with('consultRequest.medicalProblem')
        ->latest()
        ->get();

        $sicknotes = SickNote::where('consult_id', $consultId)
        // ->where('doctor_id', $doctorId)
        ->with('consultRequest.medicalProblem')
        ->latest()
        ->get();


        $allConsults = collect()
            ->merge($medicalReports)
            ->merge($prescriptions)
            ->merge($referrals)
            ->merge($sicknotes)

            ->sortByDesc('created_at');

        $recentConsult = $allConsults->first();

        

        return view('admin.add_prescription', compact(
            'patient',
            'doctor',
            'medicalReports',
            'medicalReportSingle',
            'prescriptions',
            'referrals',
            'sicknotes',
            'consult',
            'recentConsult',
            'allConsults',
            'consultRequest',
            'medical_problem'
        ));
    }


public function storeMedicalReport(Request $request)
{


    $request->validate([
        'Date_of_Consult'   => 'required|date',
        // 'Time'              => 'required',
        'Diagnosis'         => 'required|string|max:255',
        'Current_anamnesis' => 'nullable|string',
        'Treatment'         => 'required|string',
        'Plan'              => 'nullable|string',
        'Comorbidities'     => 'nullable|string',
        'Usual_Treatment'   => 'nullable|string',
        'Allergies'         => 'nullable|string',
        'email' => 'required|array',
        'email.*' => 'required|email',

    ]);

    

    $data = $request->all();
    $data['patient_id'] = $request->patient_id; // or pass patient_id from route
    $data['doctor_id'] = Auth::guard('doctor')->id();
    $data['consult_id'] = $request->consult_id;
    $data['email'] = json_encode($request->email);
    $data['type'] = 'emr';


    $report = PatientsMedicalReport::create($data);

    $patient = \App\Models\tbl_patient::findOrFail($request->patient_id);
    $doctor = Auth::guard('doctor')->user();

    // generate PDF
    $pdfName = 'sickNote_' . now()->timestamp . '.pdf';
    $savePath = public_path('sickNotePDF/' . $pdfName);

    if (!file_exists(public_path('sickNotePDF'))) {
        mkdir(public_path('sickNotePDF'), 0777, true);
    }
 
    $pdf = Pdf::loadView('pdf.medicalreport', [
        'report'    => $report,
        'patient' => $patient,
        'doctor'  => $doctor,
    ])->setPaper('A4', 'portrait');

    $pdf->setOptions([
        'isRemoteEnabled' => true,
        'isHtml5ParserEnabled' => true, // sometimes needed for https images
        'isFontSubsettingEnabled' => true
    ]);

    $pdf->save($savePath);

    $report->update(['pdf_path' => 'sickNotePDF/' . $pdfName]);

    $doctor_email = $doctor->email;
     $doctor_name = $doctor->firstName.' '.$doctor->lastName;
    
    
    foreach ($request->email as $recipient) {
        Mail::to($recipient)->send(new \App\Mail\MedicalReportMail($report,$patient,$savePath,$doctor_email,$doctor_name));
    }

    return back()->with('success', 'Medical report pdf send successfully!');
}

    /**
     * Store Prescription
     */
public function storePrescription(Request $request)
{
    $request->validate([
        'Diagnosis' => 'required|string|max:500',
                'email' => 'required|array',
    'email.*' => 'required|email',
    ]);

    

     $data = $request->all();
    $data['patient_id'] = $request->patient_id; // or pass patient_id from route
    $data['doctor_id'] = Auth::guard('doctor')->id();
    $data['consult_id'] = $request->consult_id;
    $data['email'] = json_encode($request->email);
    $data['type'] = 'emr';

        $prescription = Prescription::create($data);

    $patient = \App\Models\tbl_patient::findOrFail($request->patient_id);
    $doctor = Auth::guard('doctor')->user();

    // generate PDF
    $pdfName = 'sickNote_' . now()->timestamp . '.pdf';
    $savePath = public_path('sickNotePDF/' . $pdfName);

    if (!file_exists(public_path('sickNotePDF'))) {
        mkdir(public_path('sickNotePDF'), 0777, true);
    }

    $pdf = Pdf::loadView('pdf.prescription', [
        'prescription'    => $prescription,
        'patient' => $patient,
        'doctor'  => $doctor,
    ])->setPaper('A4', 'portrait');

    $pdf->setOptions([
        'isRemoteEnabled' => true,
        'isHtml5ParserEnabled' => true, // sometimes needed for https images
        'isFontSubsettingEnabled' => true
    ]);
    $pdf->save($savePath);

   $prescription->update(['pdf_path' => 'sickNotePDF/' . $pdfName]);

    $doctor_email = $doctor->email;
    $doctor_name = $doctor->firstName.' '.$doctor->lastName;


    // email sick note with PDF attached
      foreach ($request->email as $recipient) {
        Mail::to($recipient)->send(new \App\Mail\PrescriptionMail($prescription ,$patient,$savePath,$doctor_email,$doctor_name));
    }


    return back()->with('success', 'PDF send successfully!');
}



public function storeReferral(Request $request)
{
    $request->validate([
        'Indication' => 'required|string|max:255',
        'Requested_Referral' => 'required|string|max:255',
        'email' => 'required|array',
        'email.*' => 'required|email',
    ]);

    

    $data = $request->all();
    $data['patient_id'] = $request->patient_id; // or pass patient_id from route
    $data['doctor_id'] = Auth::guard('doctor')->id();
    $data['consult_id'] = $request->consult_id;
    $data['email'] = json_encode($request->email);
    $data['type'] = 'emr';

    $referral = PatientsRefferal::create($data);

    $patient = \App\Models\tbl_patient::findOrFail($request->patient_id);
    $doctor = Auth::guard('doctor')->user();

    // generate PDF
    $pdfName = 'sickNote_' . now()->timestamp . '.pdf';
    $savePath = public_path('sickNotePDF/' . $pdfName);

    if (!file_exists(public_path('sickNotePDF'))) {
        mkdir(public_path('sickNotePDF'), 0777, true);
    }

    $pdf = Pdf::loadView('pdf.referral', [
        'referral'    => $referral,
        'patient' => $patient,
        'doctor'  => $doctor,
    ])->setPaper('A4', 'portrait');

    $pdf->setOptions([
        'isRemoteEnabled' => true,
        'isHtml5ParserEnabled' => true, // sometimes needed for https images
        'isFontSubsettingEnabled' => true
    ]);

    $pdf->save($savePath);

    $referral->update(['pdf_path' => 'sickNotePDF/' . $pdfName]);

     $doctor_email = $doctor->email;
    $doctor_name = $doctor->firstName.' '.$doctor->lastName;

    // email sick note with PDF attached
      foreach ($request->email as $recipient) {
        Mail::to($recipient)->send(new \App\Mail\ReferralMail($referral,$patient,$savePath,$doctor_email,$doctor_name));
    }


    return back()->with('success', 'Referral report pdf send successfully!');
}

public function storeSickNote(Request $request)
{
    $validator = \Validator::make($request->all(), [
        'Reason_for_Absence' => 'required|string',
        'Start_Date_of_Absence' => 'required|date',
        'email' => 'required|array',
        'email.*' => 'required|email',
        'End_Date_of_Absence' => 'required|date|after_or_equal:Start_Date_of_Absence',
        'Return_Date' => 'required|date|after_or_equal:End_Date_of_Absence',
        
    ], [
        'End_Date_of_Absence.after_or_equal' => 'End date must be after or equal to start date.',
        'Return_Date.after_or_equal' => 'Return date must be after or equal to end date.',
    ]);

    if ($validator->fails()) {
        if ($request->ajax()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }
        return back()->withErrors($validator)->withInput();
    }

    $data = $request->all();
    
    
    $data['patient_id'] = $request->patient_id;
    $data['doctor_id'] = Auth::guard('doctor')->id();
    $data['consult_id'] = $request->consult_id;
    $data['email'] = json_encode($request->email);
    $data['type'] = 'emr';
    // dd($data);
    $note = SickNote::create($data);

    $patient = \App\Models\tbl_patient::findOrFail($request->patient_id);
    $doctor = Auth::guard('doctor')->user();

    // generate PDF
    $pdfName = 'sickNote_' . now()->timestamp . '.pdf';
    $savePath = public_path('sickNotePDF/' . $pdfName);

    if (!file_exists(public_path('sickNotePDF'))) {
        mkdir(public_path('sickNotePDF'), 0777, true);
    }

    $pdf = Pdf::loadView('pdf.sicknote', [
        'note'    => $note,
        'patient' => $patient,
        'doctor'  => $doctor,
    ])->setPaper('A4', 'portrait');

    $pdf->setOptions([
        'isRemoteEnabled' => true,
        'isHtml5ParserEnabled' => true, // sometimes needed for https images
        'isFontSubsettingEnabled' => true
    ]);

    $pdf->save($savePath);
    
    $note->update(['pdf_path' => 'sickNotePDF/' . $pdfName]);

    $doctor_email = $doctor->email;
    $doctor_name = $doctor->firstName.' '.$doctor->lastName;

    // email sick note with PDF attached
    foreach ($request->email as $recipient) {
        Mail::to($recipient)->send(new \App\Mail\SickNoteMail($note,$patient,$savePath,$doctor_email,$doctor_name));
    }

    return back()->with('success', 'Sick Note PDF generated successsfully!');
}
}